/*  -*- pse-c -*-
 *----------------------------------------------------------------------------
 * Filename: drv_alm.c
 * $Revision: 1.7 $
 *----------------------------------------------------------------------------
 * Gart and DRM driver for Intel Embedded Graphics Driver
 * Copyright  2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

#include "global.h"
#include "intelpci.h"

static int iegd_alm_configure(void);
static int iegd_alm_fetch_size(void);
static void iegd_alm_cleanup(void);
static void iegd_alm_tlbflush(struct agp_memory *mem);

static int iegd_alm_insert_entries(
	struct agp_memory *mem,
	off_t pg_start,
	int type);

static int iegd_alm_remove_entries(
	struct agp_memory *mem,
	off_t pg_start,
	int type);

struct aper_size_info_fixed intel_i830_sizes[] =
{
	{128, 32768, 5},
	/* The 64M mode still requires a 128k gatt */
	{64, 16384, 5},
	{256, 65536, 6},
	{512, 131072, 7},
};

struct aper_size_info_fixed intel_i810_sizes[] =
{
	{64, 16384, 4},
	{32, 8192,  4},
};

bridge_driver_t drv_alm = {
	.owner              = THIS_MODULE,
	.size_type          = FIXED_APER_SIZE,
	.aperture_sizes     = 0,
	.num_aperture_sizes = 0,
	.needs_scratch_page = TRUE,
	.configure          = iegd_alm_configure,
	.fetch_size         = iegd_alm_fetch_size,
	.cleanup            = iegd_alm_cleanup,
	.tlb_flush          = iegd_alm_tlbflush,
	.mask_memory        = iegd_cmn_mask_memory,
	.masks              = iegd_cmn_masks,
	.agp_enable         = iegd_cmn_agp_enable,
	.cache_flush        = global_cache_flush,
	.create_gatt_table  = NULL,
	.free_gatt_table    = NULL,
	.insert_memory      = iegd_alm_insert_entries,
	.remove_memory      = iegd_alm_remove_entries,
	.alloc_by_type      = iegd_cmn_alloc_by_type,
	.free_by_type       = iegd_cmn_free_by_type,
	.agp_alloc_page     = agp_generic_alloc_page,
	.agp_destroy_page   = agp_generic_destroy_page,
};

static int iegd_alm_configure(void)
{
	struct aper_size_info_fixed *current_size;
	u32 temp;
	u16 gmch_ctrl;
	int i;
	int entries_start = 0;

	AGN_DEBUG("Enter");

	current_size = A_SIZE_FIX(agp_bridge->current_size);

	if(private_data.pdev->device == PCI_DEVICE_ID_810   ||
	   private_data.pdev->device == PCI_DEVICE_ID_810DC ||
	   private_data.pdev->device == PCI_DEVICE_ID_810E  ||
	   private_data.pdev->device == PCI_DEVICE_ID_815) {

		pci_read_config_dword(private_data.pdev, I810_MMADDR, &temp);
		temp &= 0xfff80000;

		private_data.registers = ioremap(temp, 128*4096);
		if(!private_data.registers) {
			AGN_ERROR("Unable to remap memory");
			return -ENOMEM;
		}

		if((readl(private_data.registers+I810_DRAM_CTL)
				& I810_DRAM_ROW_0) == I810_DRAM_ROW_0_SDRAM) {
			AGN_LOG("Detected 4MB dedicated video RAM.");
			private_data.num_dcache_entries = 1024;
		}
	} else if(private_data.pdev->device == PCI_DEVICE_ID_830M ||
			  private_data.pdev->device == PCI_DEVICE_ID_845G ||
			  private_data.pdev->device == PCI_DEVICE_ID_855  ||
			  private_data.pdev->device == PCI_DEVICE_ID_865G) {

		entries_start = private_data.gtt_entries;
		pci_read_config_word(agp_bridge->dev, I830_GMCH_CTRL, &gmch_ctrl);
		gmch_ctrl |= I830_GMCH_ENABLED;
		pci_write_config_word(agp_bridge->dev,I830_GMCH_CTRL,gmch_ctrl);
	}

	/* Get based address of the graphic aperture */
	pci_read_config_dword(private_data.pdev, I810_GMADDR, &temp);
	agp_bridge->gart_bus_addr = (temp & PCI_BASE_ADDRESS_MEM_MASK);

	/* Write the based address of the gtt table to the
	 * page table control register */
	writel(agp_bridge->gatt_bus_addr | I810_PGETBL_ENABLED,
			private_data.registers+I810_PGETBL_CTL);
	readl(private_data.registers+I810_PGETBL_CTL);

	if (agp_bridge->driver->needs_scratch_page) {
		for (i = entries_start; i < current_size->num_entries; i++) {
			writel(agp_bridge->scratch_page,
					private_data.registers+I810_PTE_BASE+(i*4));
			/* PCI Posting. */
			readl(private_data.registers+I810_PTE_BASE+(i*4));
		}
	}

	global_cache_flush();

	AGN_DEBUG("Exit");
	return 0;
}


static int iegd_alm_fetch_size(void)
{
	u32 smram_miscc;
	u16 gmch_ctrl;
	struct aper_size_info_fixed *values;

	AGN_DEBUG("Enter");

	values = A_SIZE_FIX(agp_bridge->driver->aperture_sizes);

	if(private_data.pdev->device == PCI_DEVICE_ID_810   ||
	   private_data.pdev->device == PCI_DEVICE_ID_810DC ||
	   private_data.pdev->device == PCI_DEVICE_ID_810E  ||
	   private_data.pdev->device == PCI_DEVICE_ID_815) {

		pci_read_config_dword(agp_bridge->dev, I810_SMRAM_MISCC,
				&smram_miscc);

		if ((smram_miscc & I810_GMS) == I810_GMS_DISABLE) {
			printk(KERN_WARNING PFX "i810 is disabled\n");
			return 0;
		}
		if ((smram_miscc & I810_GFX_MEM_WIN_SIZE) ==
				I810_GFX_MEM_WIN_32M) {
			agp_bridge->previous_size =
				agp_bridge->current_size = (void *) (values + 1);
			agp_bridge->aperture_size_idx = 1;
			return values[1].size;
		} else {
			agp_bridge->previous_size =
				agp_bridge->current_size = (void *) (values);
			agp_bridge->aperture_size_idx = 0;
			return values[0].size;
		}
	} else if(private_data.pdev->device == PCI_DEVICE_ID_830M ||
			  private_data.pdev->device == PCI_DEVICE_ID_845G ||
			  private_data.pdev->device == PCI_DEVICE_ID_855  ||
			  private_data.pdev->device == PCI_DEVICE_ID_865G) {

		if (agp_bridge->dev->device != PCI_DEVICE_ID_INTEL_82830_HB &&
			agp_bridge->dev->device != PCI_DEVICE_ID_INTEL_82845G_HB) {
			/* 855GM/852GM/865G has 128MB aperture size */
			agp_bridge->previous_size =
				agp_bridge->current_size = (void *) values;
			agp_bridge->aperture_size_idx = 0;
			return values[0].size;
		}

		pci_read_config_word(agp_bridge->dev,I830_GMCH_CTRL,&gmch_ctrl);

		if ((gmch_ctrl & I830_GMCH_MEM_MASK) == I830_GMCH_MEM_128M) {
			agp_bridge->previous_size =
				agp_bridge->current_size = (void *) values;
			agp_bridge->aperture_size_idx = 0;
			return values[0].size;
		} else {
			agp_bridge->previous_size =
				agp_bridge->current_size = (void *) (values + 1);
			agp_bridge->aperture_size_idx = 1;
			return values[1].size;
		}
	}

	AGN_DEBUG("Exit");

	return 0;
}

static void iegd_alm_cleanup(void)
{

	AGN_DEBUG("Enter");

	if(private_data.pdev->device == PCI_DEVICE_ID_810   ||
	   private_data.pdev->device == PCI_DEVICE_ID_810DC ||
	   private_data.pdev->device == PCI_DEVICE_ID_810E  ||
	   private_data.pdev->device == PCI_DEVICE_ID_815) {

		writel(0, private_data.registers+I810_PGETBL_CTL);
		readl(private_data.registers);	/* PCI Posting. */
	}

	/* Unmap the mapping of the mmio */
	iounmap((void *) private_data.registers);

	AGN_DEBUG("Exit");
}

static void iegd_alm_tlbflush(struct agp_memory *mem)
{
	AGN_DEBUG("Enter");
	return;
	AGN_DEBUG("Exit");
}

int AGP_CREATE_GATT(iegd_alm_create_gatt_table)
{
	int num_entries         = 0;
	int i830_gtt_page_order = 0;
	u32 gtt_bus_addr        = 0;
	u32 mmio_bus_addr       = 0;
	char *gtt_table         = NULL;
	char *gtt_table_end     = NULL;
	char *current_entry     = NULL;
	int gtt_enabled         = FALSE;
	struct page *gtt_table_page            = NULL;
	struct aper_size_info_fixed *aper_size = NULL;

	AGN_DEBUG("Enter");

	agp_bridge->gatt_table_real = NULL;
	agp_bridge->gatt_table      = NULL;
	aper_size = (struct aper_size_info_fixed *)agp_bridge->current_size;

	/* Find and save the address of the MMIO registers */
	pci_read_config_dword(private_data.pdev, I810_MMADDR,
		&mmio_bus_addr);
	mmio_bus_addr &= 0xFFF80000;

	private_data.registers = (volatile u8 *)
		ioremap(mmio_bus_addr, 128 * 4096);

	if (!private_data.registers) {
		AGN_ERROR("ioremap failed to map");
		return (-ENOMEM);
	}

	/* Get value on the control register */
	gtt_bus_addr = readl(private_data.registers+I810_PGETBL_CTL) &
		0xFFFFF000;
	gtt_enabled = readl(private_data.registers+I810_PGETBL_CTL) &
		I810_PGETBL_ENABLED;
	global_cache_flush();

	/* we have to call this as early as possible after the MMIO base address
	 * is known */
	iegd_cmn_init_gtt_entries();

	/* If GTT does not exist, which can happen if a PCI graphics card is the
	 * boot-up display device, then we will have to allocate the GTT table
	 * ourselves
	 */
	if (!gtt_enabled) {

		AGN_DEBUG("Gtt is disabled");

		i830_gtt_page_order = aper_size->page_order;
		num_entries = aper_size->num_entries;
		gtt_table = (char *) __get_free_pages(
				GFP_KERNEL, i830_gtt_page_order);
		gtt_table_end = gtt_table +
			((PAGE_SIZE * (1<<i830_gtt_page_order)) - 1);

		/* Make sure allocation was successful */
		if (NULL == gtt_table) {
			AGN_ERROR("Fail to allocate kernel memory");
			return -ENOMEM;
		}

		for (current_entry = gtt_table; current_entry < gtt_table_end;
			current_entry += PAGE_SIZE) {
			gtt_table_page = virt_to_page(current_entry);
			set_bit(PG_reserved, &gtt_table_page->flags);
		}
		agp_bridge->gatt_bus_addr = virt_to_phys(gtt_table);
	} else {
		agp_bridge->gatt_bus_addr = gtt_bus_addr;
	}

	AGN_DEBUG("Exit");
	return(0);
}


static int iegd_alm_insert_entries(
	struct agp_memory *mem,
	off_t pg_start,
	int type)
{
	int i, j, num_entries;
	void *temp;

	AGN_DEBUG("Enter");

	temp = agp_bridge->current_size;
	num_entries = A_SIZE_FIX(temp)->num_entries;

	if ((pg_start + mem->page_count) > num_entries) {
		AGN_ERROR("Trying to write beyond aperture limit");
		AGN_DEBUG("pg_start=0x%.8lx, mem->page_count=%d,"
				"num_entries=%d", pg_start, mem->page_count,
				num_entries);
		return -EINVAL;
	}

	if(private_data.pdev->device == PCI_DEVICE_ID_830M ||
	   private_data.pdev->device == PCI_DEVICE_ID_845G ||
	   private_data.pdev->device == PCI_DEVICE_ID_855  ||
	   private_data.pdev->device == PCI_DEVICE_ID_865G) {

		if (pg_start < private_data.gtt_entries) {
			AGN_ERROR("Trying to insert into local/stolen memory");
			AGN_DEBUG("pg_start == 0x%.8lx,private_data.gtt_entries =="
				"0x%.8x", pg_start,private_data.gtt_entries);
			return -EINVAL;
		}
	} else if(private_data.pdev->device == PCI_DEVICE_ID_810   ||
		      private_data.pdev->device == PCI_DEVICE_ID_810DC ||
		      private_data.pdev->device == PCI_DEVICE_ID_810E  ||
		      private_data.pdev->device == PCI_DEVICE_ID_815) {

		for (j = pg_start; j < (pg_start + mem->page_count); j++) {
			if (!PGE_EMPTY(agp_bridge, readl(agp_bridge->gatt_table+j))) {
				AGN_ERROR("Device busy");
				return -EBUSY;
			}
		}
		if (type != 0 || mem->type != 0) {
			if ((type == AGP_DCACHE_MEMORY) &&
				(mem->type == AGP_DCACHE_MEMORY)) {
				/* special insert */
				global_cache_flush();
				for (i = pg_start; i < (pg_start + mem->page_count);
					i++) {
					writel((i*4096)|I810_PTE_LOCAL|I810_PTE_VALID,
						private_data.registers+I810_PTE_BASE+(i*4));
					/* PCI Posting. */
					readl(private_data.registers +
							I810_PTE_BASE+(i*4));
				}
				global_cache_flush();
				agp_bridge->driver->tlb_flush(mem);
				AGN_DEBUG("AGP_DCACHE_MEMORY.. Exit");
				return 0;
			}
		}
	}

	if ((type != 0 && type != AGP_PHYS_MEMORY) ||
		(mem->type != 0 && mem->type != AGP_PHYS_MEMORY)) {
		AGN_ERROR("Unsupported memory type");
		AGN_DEBUG("mem->type=%x", mem->type);
		return -EINVAL;
	}

	global_cache_flush();
	for (i = 0, j = pg_start; i < mem->page_count; i++, j++) {
		writel(AGP_MASK_GTT(), private_data.registers+I810_PTE_BASE+(j*4));
		/* PCI Posting. */
		readl(private_data.registers+I810_PTE_BASE+(j*4));
	}

	global_cache_flush();
	agp_bridge->driver->tlb_flush(mem);

	AGN_DEBUG("Exit");

	return 0;
}

static int iegd_alm_remove_entries(
	struct agp_memory *mem,
	off_t pg_start,
	int type)
{
	int i;

	AGN_DEBUG("Enter");

	global_cache_flush();

	if(private_data.pdev->device == PCI_DEVICE_ID_830M ||
	   private_data.pdev->device == PCI_DEVICE_ID_845G ||
	   private_data.pdev->device == PCI_DEVICE_ID_855  ||
	   private_data.pdev->device == PCI_DEVICE_ID_865G) {

		if (pg_start < private_data.gtt_entries) {
			AGN_ERROR("Trying to disable local/stolen memory");
			AGN_DEBUG("pg_start=0x%.8lx, private_data.gtt_entries=%d",
					pg_start, private_data.gtt_entries);
			return -EINVAL;
		}
	}

	for (i = pg_start; i < (mem->page_count + pg_start); i++) {
		writel(agp_bridge->scratch_page,
			private_data.registers+I810_PTE_BASE+(i*4));
		/* PCI Posting. */
		readl(private_data.registers+I810_PTE_BASE+(i*4));
	}

	global_cache_flush();
	agp_bridge->driver->tlb_flush(mem);

	AGN_DEBUG("Exit");

	return 0;
}

